<?php

namespace App\Http\Controllers\app;

use App\Http\Controllers\Controller;
use App\Http\Helpers\Helper;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;

class AdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('is_verify_email');
    }

    public function list()
    {
        $title = "Admin List";
        $adminList = Admin::get();
        return view('app.admin.admin_list',compact('adminList','title'));

    }
    public function create()
    {
        $key = "Create";
        $title = "Add Admin";
        return view('app.admin.admin_form',compact('key','title'));
    }

    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'name' => 'required|min:3|max:30',
            'email_id' => 'required|min:3|email|unique:admins,email,',
            'username' => 'required|min:6|unique:admins',
            'phone_number'=>'required|numeric|unique:admins',
            'password' => 'required|min:8|max:30',
        ]);

        $adminInput = new Admin;
        $adminInput->name = $validatedData['name'];
        $adminInput->email = $validatedData['email_id'];
        $adminInput->username = $validatedData['username'];
        $adminInput->password = Hash::make($validatedData['password']);
        $adminInput->phone_number = isset($request->phone_number)?$request->phone_number:'';

        if ($request->hasFile('profile_image')) {
//            $adminInput->profile_image = uploadFile($request->profile_image, 'admin', 'uploads/admin/profile_image/');
            $adminInput->profile_image = uploadFile($request->profile_image, 'uploads/admin/profile_image/', 'admin');

        }

        $usernameData = Admin::where('username',$validatedData['username'])->count();

        if($usernameData==0){
            $emailData = Admin::where('email',$validatedData['email_id'])->count();
            if($emailData==0){
                $token = Str::random(64);
                $adminInput->token = $token;
                if($adminInput->save()){
                    Mail::send('app.auth.verify', ['token' => $token], function($message) use($request){
                        $message->to($request->email_id);
                        $message->subject('Email Verification Mail');
                    });
                    session()->flash('success', 'Admin "'.$adminInput->name.'" has been added successfully');
                    return redirect(sitePrefix().'administration');
                } else{
                    return back()->with('message', 'Error while creating admin');
                }
            } else{
                return back()->withInput($request->input())->withErrors("E-mail ID '".$validatedData['email_id']."' has been already taken");
            }
        } else{
            return back()->withInput($request->input())->withErrors("Username '".$validatedData['username']."' has been already taken");
        }
    }


    public function verifyAccount($token)
    {
        $verifyUser = Admin::where('token', $token)->first();

        $message = 'Sorry your email cannot be identified.';

        if (!is_null($verifyUser)) {
            if (!($verifyUser->email_verified_at)) {
                $verifyUser->email_verified_at = now();
                $verifyUser->save();
                $message = "Your e-mail is verified. You can now login.";
            } else {
                $message = "Your e-mail is already verified. You can now login.";
            }
        }
        return redirect()->route('login')->with('message', $message);
    }

    public function edit(Request $request, $id)
    {
        $key = "Update";
        $title = "Admin Update";
        $admin = Admin::find($id);
        if($admin!=NULL){
            return view('app.admin.admin_form', compact('key','admin','title'));
        } else{
            return view('app.errors.404');
        }
    }

    public function view(Request $request, $id)
    {
        $key = "View";
        $admin = Admin::find($id);
        if($admin!=NULL){
            $title = "Admin View - ".$admin->name;
            return view('app.admin.admin_view', compact('key','admin','title'));
        } else{
            return view('app.errors.404');
        }
    }

    public function update(Request $request, $id)
    {
        $validatedData = $request->validate([
            'name' => 'required|min:3|max:30',
            'email_id' => 'required|min:3|email|unique:admins,email,'.$id,
            'username'=>'required|min:6|max:30|unique:admins,username,'.$id,
            'phone_number'=>'required|numeric|unique:admins,phone_number,'.$id,
        ]);

        $adminInput = Admin::find($id);
        $adminInput->name = $validatedData['name'];
        $adminInput->email = $validatedData['email_id'];
        $adminInput->phone_number = isset($request->phone_number)?$request->phone_number:'';

        if ($request->hasFile('profile_image')) {
            if (File::exists($adminInput->profile_image)) {
                File::delete($adminInput->profile_image);
            }
            $adminInput->profile_image = uploadFile($request->profile_image, 'uploads/admin/profile_image/', 'admin');
        }

        $emailData = Admin::where([['email',$validatedData['email_id']],['id','!=',$id]])->count();
        if($emailData==0){
            $usernameData = Admin::where([['username',$validatedData['username']],['id','!=',$id]])->count();
            if($usernameData==0){
                $adminInput->username = $validatedData['username'];
                if($adminInput->save()){
                    session()->flash('success', 'Admin "'.$adminInput->name.'" has been updated successfully');
                    return redirect(sitePrefix().'administration');
                } else{
                    return back()->with('message', 'Error while updating admin');
                }
            } else{
                return back()->withInput($request->input())->withErrors("E-mail ID '".$validatedData['email_id']."' has been already taken");
            }
        } else{
            return back()->withInput($request->input())->withErrors("E-mail ID '".$validatedData['email_id']."' has been already taken");
        }
    }

    public function reset_password($id)
    {
        $admin = Admin::find($id);
        if($admin!=NULL){
            $key = "Reset Password - ".$admin->name;
            $title = "Reset Password - ".$admin->name;
            return view('app.admin.reset_password',compact('key','title','admin'));
        } else{
            return view('app.errors.404');
        }
    }

    public function reset_password_store(Request $request ,$id)
    {
        $validatedData = $request->validate([
            'password' => 'required|min:8',
            'confirm_password' => 'required|min:8',
        ]);

        if($validatedData['password']==$validatedData['confirm_password']){
            $admin = Admin::find($id);
            $admin->password = Hash::make($validatedData['password']);
            if($admin->save()){
                session()->flash('success', 'Password has been updated successfully');
                return redirect(sitePrefix() . 'dashbard/reset_password/' . $id);
            } else{
                return back()->with('message', 'Error while resetting the password');
            }
        } else{
            return back()->withInput($request->input())->withErrors("Password mis-match");
        }
    }

    public function delete_admin(Request $request){
        if(isset($request->id) && $request->id!=NULL){
            $admin = Admin::find($request->id);
            if($admin){
                if($admin->id==1){
                    return response()->json([
                        'status' => false,
                        'message' => "Not permitted to delete the 'Super Admin' user"
                    ]);
                } else{
                    $deleted = $admin->delete();
//                    if (File::exists($admin->profile_image)) {
//                        File::delete($admin->profile_image);
//                    }
                    if($deleted==true){
                        echo(json_encode(array('status'=>true)));
                    } else{
                        echo(json_encode(array('status'=>false,'message'=>'Some error occurred,please try after sometime')));
                    }
                }
            } else{
                echo(json_encode(array('status'=>false,'message'=>'Model class not found')));
            }
        }
    }


    public function profile()
    {
        $adminData = Auth::user();
        if ($adminData) {
            return view('app.admin.profile', compact('adminData'));
        } else {
            return view('backend.error.404');
        }
    }


    public function profile_store(Request $request)
    {

        $admin = Auth::user();
        $validatedData = $request->validate([
            'name' => 'required|min:3|max:30',
            'email_id' => 'required|min:3|email|unique:admins,email,'.$admin->id,
            'username'=>'required|min:5|max:30|unique:admins,username,'.$admin->id,
            'phone_number'=>'required|numeric|unique:admins,phone_number,'.$admin->id,
        ]);

        $admin->name = $validatedData['name'];
        $admin->email = $validatedData['email_id'];
        $admin->phone_number = isset($request->phone_number)?$request->phone_number:'';

        if ($request->hasFile('profile_image')) {
            if (File::exists($admin->profile_image)) {
                File::delete($admin->profile_image);
            }
            $admin->profile_image = uploadFile($request->profile_image, 'uploads/admin/profile_image/', 'admin');
        }
        $emailData = Admin::where([['email',$validatedData['email_id']],['id','!=',$admin->id]])->count();
        if($emailData==0){
            $usernameData = Admin::where([['username',$validatedData['username']],['id','!=',$admin->id]])->count();
            if($usernameData==0){
                $admin->username = $validatedData['username'];
                if($admin->save()){
                    session()->flash('success', 'Admin "'.$admin->name.'" has been updated successfully');
                    return redirect(sitePrefix().'administration');
                } else{
                    return back()->with('message', 'Error while updating profile');
                }
            } else{
                return back()->withInput($request->input())->withErrors("E-mail ID '".$validatedData['email_id']."' has been already taken");
            }
        } else{
            return back()->withInput($request->input())->withErrors("E-mail ID '".$validatedData['email_id']."' has been already taken");
        }



    }


}
