<script>
// Security features
document.addEventListener('contextmenu', function(e) { e.preventDefault(); return false; });
document.addEventListener('keydown', function(e) {
  if (e.key === 'F12' || (e.ctrlKey && e.shiftKey && e.key === 'I') || (e.ctrlKey && e.shiftKey && e.key === 'C') || (e.ctrlKey && e.key === 'U')) {
    e.preventDefault(); return false;
  }
});
document.addEventListener('dragstart', function(e) { if (e.target.tagName === 'IMG') { e.preventDefault(); return false; } });

/* -------------------------
   Multilingual Support
   ------------------------- */
const translations = {
  en: {
    dashboardTitle: "📋 Client Enquiry Dashboard",
    databaseStatus: "Database Connected / Live",
    refreshButton: " Refresh",
    exportPDFButton: " Export PDF",
    darkLightButton: " Dark/Light",
    autoAssignButton: " Auto-Assign",
    totalEnquiries: "Total Enquiries",
    todayEnquiries: "Enquiries Today",
    unassignedEnquiries: "Unassigned",
    tableId: "#",
    tableName: "Name",
    tablePhone: "Phone",
    tableMessage: "Message",
    loadingText: "Loading...",
    closeButton: "Close",
    noEnquiriesText: "No enquiries",
    errorLoadingText: "Error loading data",
    filterByName: "Filter by Name",
    filterByPhone: "Filter by Phone",
    filterByDate: "Filter by Date",
    rowsPerPage: "Rows per page",
    newEnquiry: "New Enquiry",
    assignAgentTitle: "Assign Enquiry to Agent",
    selectAgentLabel: "Select Agent", 
    enquiryDetailsLabel: "Enquiry Details",
    customMessageLabel: "Additional Notes (Optional)",
    sendToAgentButton: "Send via Telegram",
    copyMessageButton: "Copy Message"
  },
  ar: { /* Arabic translations */ },
  hi: { /* Hindi translations */ },
  tl: { /* Tagalog translations */ },
  ur: { /* Urdu translations */ }
};

let currentLanguage = 'en';

function setLanguage(lang) {
  currentLanguage = lang;
  document.querySelectorAll('[data-i18n]').forEach(element => {
    const key = element.getAttribute('data-i18n');
    if (translations[lang] && translations[lang][key]) {
      element.textContent = translations[lang][key];
    }
  });
  document.getElementById('languageSelect').value = lang;
  if (lang === 'ar') {
    document.documentElement.dir = 'rtl';
    document.body.classList.add('rtl');
  } else {
    document.documentElement.dir = 'ltr';
    document.body.classList.remove('rtl');
  }
}

document.addEventListener('DOMContentLoaded', function() {
  const savedLanguage = localStorage.getItem('dashboardLanguage') || 'en';
  setLanguage(savedLanguage);
  document.getElementById('languageSelect').addEventListener('change', function(e) {
    const newLang = e.target.value;
    setLanguage(newLang);
    localStorage.setItem('dashboardLanguage', newLang);
  });
});

/* -------------------------
   State & initialization
   ------------------------- */
let lastIds = [];
let allEnquiries = [];
let filteredEnquiries = [];
let currentPage = 1, perPage = 30;
let pageFocused = true;
let initialized = false;

// Enhanced agent management
const agents = [
  { name: "Agent Ahmed", telegramId: "123456789", department: "Sales", active: true },
  { name: "Agent Maria", telegramId: "987654321", department: "Support", active: true },
  { name: "Agent John", telegramId: "555666777", department: "Technical", active: true },
  { name: "Agent Sarah", telegramId: "444333222", department: "VIP Clients", active: true }
];

let currentAgentIndex = 0;
let autoAssignEnabled = false;
let assignedEnquiries = JSON.parse(localStorage.getItem('assignedEnquiries') || '{}');
let currentEnquiry = null;

// Telegram configuration
const telegramConfig = {
  botToken: "YOUR_BOT_TOKEN_HERE", // Replace with actual bot token
  enabled: true
};

// Saved contacts
let savedContacts = JSON.parse(localStorage.getItem('savedContacts') || '[]');

// Automation settings
let automationSettings = JSON.parse(localStorage.getItem('automationSettings') || '{}');

// Track visibility
document.addEventListener('visibilitychange', ()=> { pageFocused = document.visibilityState === 'visible'; });

/* Unlock audio on first user gesture */
document.addEventListener('click', function unlockAudio(){
  const sound = document.getElementById('notificationSound');
  if(!sound) return;
  sound.play().then(()=>{ sound.pause(); sound.currentTime = 0; document.removeEventListener('click', unlockAudio); }).catch(()=>{});
});

/* -------------------------
   UI Utilities
   ------------------------- */
function showBootstrapToast(msg, type='success'){
  const id = 't' + Date.now();
  const html = `<div id="${id}" class="toast align-items-center text-bg-${type} border-0 mb-2" role="alert">
      <div class="d-flex">
        <div class="toast-body">${msg}</div>
        <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
      </div>
    </div>`;
  $('.toast-container').append(html);
  const t = new bootstrap.Toast(document.getElementById(id), { delay: 3000 });
  t.show();
  document.getElementById(id).addEventListener('hidden.bs.toast', ()=>{ $('#' + id).remove(); });
}

function showNewEnquiryNotification(){
  const id = 'newEnq' + Date.now();
  const notif = document.createElement('div');
  notif.id = id;
  notif.className = 'new-enquiry-notif';
  notif.innerHTML = `<div class="dot" aria-hidden></div><div style="font-weight:700;color:var(--notif-text)">${translations[currentLanguage].newEnquiry || 'New Enquiry'}</div>`;
  document.body.appendChild(notif);
  setTimeout(()=>{ notif.style.transition = 'opacity .25s'; notif.style.opacity = '0'; setTimeout(()=>notif.remove(),300); }, 3000);
}

/* -------------------------
   Saved Contacts Management
   ------------------------- */
function loadSavedContacts() {
  const contactsList = $('#savedContactsList');
  const telegramSelect = $('#telegramContact');
  
  contactsList.empty();
  telegramSelect.empty();
  telegramSelect.append('<option value="">-- Select Saved Contact --</option>');
  
  if (savedContacts.length === 0) {
    contactsList.html('<p class="text-muted text-center">No saved contacts yet.</p>');
    return;
  }
  
  savedContacts.forEach((contact, index) => {
    // Add to contacts list
    const contactHtml = `
      <div class="saved-contact-item" data-index="${index}">
        <div class="d-flex justify-content-between align-items-center">
          <div>
            <strong>${contact.name}</strong>
            ${contact.designation ? `<span class="text-muted ms-2">- ${contact.designation}</span>` : ''}
            <div class="phone-number">${contact.phone}</div>
            ${contact.telegramChatId ? `<small class="text-info">Telegram: ${contact.telegramChatId}</small>` : ''}
          </div>
          <div>
            <button class="btn btn-sm btn-outline-danger delete-contact" data-index="${index}">
              <i class="bi bi-trash"></i>
            </button>
          </div>
        </div>
      </div>
    `;
    contactsList.append(contactHtml);
    
    // Add to Telegram select
    telegramSelect.append(`<option value="${contact.telegramChatId || ''}" data-phone="${contact.phone}">${contact.name} - ${contact.phone}</option>`);
  });
}

function saveContact(contact) {
  savedContacts.push(contact);
  localStorage.setItem('savedContacts', JSON.stringify(savedContacts));
  loadSavedContacts();
  showBootstrapToast('Contact saved successfully!', 'success');
}

function deleteContact(index) {
  savedContacts.splice(index, 1);
  localStorage.setItem('savedContacts', JSON.stringify(savedContacts));
  loadSavedContacts();
  showBootstrapToast('Contact deleted!', 'warning');
}

/* -------------------------
   Telegram Integration
   ------------------------- */
async function sendTelegramMessage(chatId, message) {
  if (!telegramConfig.botToken || telegramConfig.botToken === "YOUR_BOT_TOKEN_HERE") {
    showBootstrapToast('Please configure Telegram bot token first!', 'danger');
    return false;
  }
  
  try {
    const response = await fetch(`https://api.telegram.org/bot${telegramConfig.botToken}/sendMessage`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        chat_id: chatId,
        text: message,
        parse_mode: 'HTML'
      })
    });
    
    const result = await response.json();
    return result.ok;
  } catch (error) {
    console.error('Telegram API error:', error);
    return false;
  }
}

async function sendTelegramDocument(chatId, fileBlob, filename, caption = '') {
  if (!telegramConfig.botToken || telegramConfig.botToken === "YOUR_BOT_TOKEN_HERE") {
    showBootstrapToast('Please configure Telegram bot token first!', 'danger');
    return false;
  }
  
  const formData = new FormData();
  formData.append('chat_id', chatId);
  formData.append('document', fileBlob, filename);
  formData.append('caption', caption);
  
  try {
    const response = await fetch(`https://api.telegram.org/bot${telegramConfig.botToken}/sendDocument`, {
      method: 'POST',
      body: formData
    });
    
    const result = await response.json();
    return result.ok;
  } catch (error) {
    console.error('Telegram document error:', error);
    return false;
  }
}

/* -------------------------
   Chart Initialization
   ------------------------- */
function initializeCharts() {
  // Service Type Chart
  const serviceCtx = document.getElementById('serviceChart').getContext('2d');
  new Chart(serviceCtx, {
    type: 'doughnut',
    data: {
      labels: ['Property Consultation', 'Investment Advice', 'Market Analysis', 'Other'],
      datasets: [{
        data: [45, 30, 20, 5],
        backgroundColor: ['#4e73df', '#1cc88a', '#36b9cc', '#f6c23e']
      }]
    },
    options: {
      responsive: true,
      plugins: {
        legend: { position: 'bottom' }
      }
    }
  });

  // Timeline Chart
  const timelineCtx = document.getElementById('timelineChart').getContext('2d');
  new Chart(timelineCtx, {
    type: 'line',
    data: {
      labels: ['Day 1', 'Day 2', 'Day 3', 'Day 4', 'Day 5', 'Day 6', 'Day 7'],
      datasets: [{
        label: 'Enquiries',
        data: [12, 19, 8, 15, 12, 16, 11],
        borderColor: '#4e73df',
        backgroundColor: 'rgba(78, 115, 223, 0.1)',
        tension: 0.4,
        fill: true
      }]
    },
    options: {
      responsive: true,
      scales: {
        y: { beginAtZero: true }
      }
    }
  });

  // Revenue Chart
  const revenueCtx = document.getElementById('revenueChart').getContext('2d');
  new Chart(revenueCtx, {
    type: 'bar',
    data: {
      labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun'],
      datasets: [{
        label: 'Revenue ($)',
        data: [12500, 19000, 15300, 17800, 19200, 21500],
        backgroundColor: '#1cc88a'
      }]
    },
    options: {
      responsive: true,
      scales: {
        y: { beginAtZero: true }
      }
    }
  });

  // Agent Performance Chart
  const agentCtx = document.getElementById('agentChart').getContext('2d');
  new Chart(agentCtx, {
    type: 'polarArea',
    data: {
      labels: ['Ahmed', 'Maria', 'John', 'Sarah'],
      datasets: [{
        data: [85, 78, 92, 88],
        backgroundColor: ['#4e73df', '#1cc88a', '#36b9cc', '#f6c23e']
      }]
    },
    options: {
      responsive: true
    }
  });
}

/* -------------------------
   Enhanced PDF Generation with Header
   ------------------------- */
function generatePDF(data, filename) {
  const { jsPDF } = window.jspdf;
  const doc = new jsPDF();
  
  // Add header with logo and company name
  doc.setFontSize(20);
  doc.setTextColor(40, 40, 40);
  doc.text('Mountain Gate L.L.C', 105, 20, { align: 'center' });
  doc.setFontSize(16);
  doc.setTextColor(100, 100, 100);
  doc.text('Enquiry Report', 105, 30, { align: 'center' });
  
  // Add date
  doc.setFontSize(10);
  doc.setTextColor(150, 150, 150);
  doc.text(`Generated on: ${new Date().toLocaleDateString()}`, 105, 38, { align: 'center' });
  
  // Add table
  const rows = data.map((r,i) => [
    i+1, 
    r.name || '', 
    r.phone || '', 
    r.service || 'N/A',
    (r.requirement || '').substring(0, 50) + '...',
    new Date(r.created_at).toLocaleDateString(),
    r.status || 'Pending'
  ]);
  
  doc.autoTable({
    startY: 45,
    head: [['#', 'Name', 'Phone', 'Service', 'Message', 'Date', 'Status']],
    body: rows,
    styles: { fontSize: 8 },
    headStyles: { fillColor: [78, 115, 223] }
  });
  
  doc.save(filename);
}

/* -------------------------
   Excel/CSV Export
   ------------------------- */
function exportToExcel(data, filename) {
  const ws = XLSX.utils.json_to_sheet(data.map(item => ({
    'Name': item.name,
    'Phone': item.phone,
    'Service': item.service,
    'Message': item.requirement,
    'Date': new Date(item.created_at).toLocaleDateString(),
    'Status': item.status
  })));
  
  const wb = XLSX.utils.book_new();
  XLSX.utils.book_append_sheet(wb, ws, 'Enquiries');
  XLSX.writeFile(wb, `${filename}.xlsx`);
}

function exportToCSV(data, filename) {
  const headers = ['Name', 'Phone', 'Service', 'Message', 'Date', 'Status'];
  const csvData = data.map(item => [
    item.name,
    item.phone,
    item.service,
    item.requirement,
    new Date(item.created_at).toLocaleDateString(),
    item.status
  ]);
  
  const csvContent = [headers, ...csvData]
    .map(row => row.map(field => `"${field}"`).join(','))
    .join('\n');
  
  const blob = new Blob([csvContent], { type: 'text/csv' });
  const url = window.URL.createObjectURL(blob);
  const a = document.createElement('a');
  a.href = url;
  a.download = `${filename}.csv`;
  a.click();
  window.URL.revokeObjectURL(url);
}

/* -------------------------
   Fetch & render
   ------------------------- */
function fetchEnquiries(){
  $.ajax({
    url: 'fetch_enquiries.php',
    dataType: 'json',
    success: function(res){
      const ids = res.map(r => r.id);
      const newIds = ids.filter(id => !lastIds.includes(id));

      if(newIds.length > 0 && initialized){
        const sound = document.getElementById('notificationSound');
        if(sound){
          try{
            sound.currentTime = 0;
            sound.play().catch(e => {
              console.warn('Audio play blocked:',e);
              playFallbackSound();
            });
          }catch(e){ 
            console.warn('Audio error', e);
            playFallbackSound();
          }
        }
        
        showNewEnquiryNotification();
        showBootstrapToast(`${newIds.length} new enquiry(s) received`, 'success');
        
        // Auto-assign if enabled
        if (autoAssignEnabled && newIds.length > 0) {
          autoAssignNewEnquiries(res.filter(r => newIds.includes(r.id)));
        }
      }

      allEnquiries = res;
      applyFilters();
      renderStats();
      renderTable();
      updateCharts();

      lastIds = ids;
      if(!initialized) initialized = true;
    },
    error: function(){
      $('#enquiriesBody').html('<tr><td colspan="8">' + (translations[currentLanguage].errorLoadingText || 'Error loading data') + '</td></tr>');
      showBootstrapToast(translations[currentLanguage].errorLoadingText || 'Unable to load enquiries', 'danger');
    }
  });
}

function playFallbackSound() {
  try {
    const audioContext = new (window.AudioContext || window.webkitAudioContext)();
    const oscillator = audioContext.createOscillator();
    const gainNode = audioContext.createGain();
    
    oscillator.connect(gainNode);
    gainNode.connect(audioContext.destination);
    
    oscillator.frequency.value = 800;
    oscillator.type = 'sine';
    
    gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
    gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.5);
    
    oscillator.start(audioContext.currentTime);
    oscillator.stop(audioContext.currentTime + 0.5);
  } catch (e) {
    console.warn('Web Audio API not supported');
  }
}

function applyFilters() {
  const nameFilter = document.getElementById('nameFilter').value.toLowerCase();
  const phoneFilter = document.getElementById('phoneFilter').value.toLowerCase();
  const dateFilter = document.getElementById('dateFilter').value;
  const statusFilter = document.getElementById('statusFilter').value;
  const serviceFilter = document.getElementById('serviceFilter').value;
  
  filteredEnquiries = allEnquiries.filter(enquiry => {
    if (nameFilter && !enquiry.name.toLowerCase().includes(nameFilter)) return false;
    if (phoneFilter && !enquiry.phone.toLowerCase().includes(phoneFilter)) return false;
    if (dateFilter) {
      const enquiryDate = new Date(enquiry.created_at).toISOString().split('T')[0];
      if (enquiryDate !== dateFilter) return false;
    }
    if (statusFilter && enquiry.status !== statusFilter) return false;
    if (serviceFilter && enquiry.service !== serviceFilter) return false;
    return true;
  });
  
  currentPage = 1;
}

function renderStats(){
  $('#totalEnquiries').text(allEnquiries.length);
  let today = 0; 
  let unassigned = 0;
  const now = new Date().toDateString();
  
  allEnquiries.forEach(e => { 
    if(new Date(e.created_at).toDateString() === now) today++; 
    if(!assignedEnquiries[e.id]) unassigned++;
  });
  
  $('#todayEnquiries').text(today);
  $('#unassignedEnquiries').text(unassigned);
  
  // Calculate conversion rate (example)
  const completed = allEnquiries.filter(e => e.status === 'Completed').length;
  const conversionRate = allEnquiries.length > 0 ? Math.round((completed / allEnquiries.length) * 100) : 0;
  $('#conversionRate').text(conversionRate + '%');
  
  // Update BI metrics
  $('#totalRevenue').text('$' + (allEnquiries.length * 250).toLocaleString());
  $('#avgResponseTime').text('15m');
  $('#clientSatisfaction').text('92%');
  $('#agentPerformance').text('88%');
}

function renderTable(){
  const start = (currentPage - 1) * perPage;
  const paginated = filteredEnquiries.slice(start, start + perPage);
  let html = '';
  
  if(paginated.length){
    paginated.forEach((row, i) => {
      const isNew = !lastIds.includes(row.id) ? 'new-row' : '';
      const index = start + i + 1;
      
      html += `<tr class="${isNew}">
        <td>${index}</td>
        <td>${escapeHtml(row.name || '')}</td>
        <td><span class="phone-number">${escapeHtml(row.phone || '')}</span></td>
        <td>${escapeHtml(row.service || 'N/A')}</td>
        <td><span class="message-content" title="${escapeHtml(row.requirement || '')}">${escapeHtml(row.requirement || '')}</span></td>
        <td>${new Date(row.created_at).toLocaleString()}</td>
        <td><span class="badge bg-${getStatusColor(row.status)}">${row.status || 'Pending'}</span></td>
        <td>
          <div class="action-buttons">
            <button class="btn btn-sm btn-outline-info copy-message-btn" title="Copy Message" data-message="${escapeHtml(row.requirement || '')}">
              <i class="bi bi-clipboard"></i>
            </button>
            <button class="btn btn-sm assign-agent-btn" title="Assign to Agent" data-enquiry-id="${row.id}">
              <i class="bi bi-person-plus"></i>
            </button>
            <button class="btn btn-sm btn-outline-success" title="Mark Complete" data-enquiry-id="${row.id}">
              <i class="bi bi-check"></i>
            </button>
          </div>
        </td>
      </tr>`;
    });
  } else {
    html = `<tr><td colspan="8">${translations[currentLanguage].noEnquiriesText || 'No enquiries'}</td></tr>`;
  }
  $('#enquiriesBody').html(html);
  renderPagination();
}

function getStatusColor(status) {
  const colors = {
    'Pending': 'warning',
    'Assigned': 'info',
    'Completed': 'success'
  };
  return colors[status] || 'secondary';
}

function renderPagination(){
  const total = Math.max(1, Math.ceil(filteredEnquiries.length / perPage));
  let html = '';
  for(let i = 1; i <= total; i++){
    html += `<li class="page-item ${i === currentPage ? 'active' : ''}"><a class="page-link" href="#">${i}</a></li>`;
  }
  $('#pagination').html(html);
  $('#pagination .page-link').click(function(e){ e.preventDefault(); currentPage = parseInt($(this).text()); renderTable(); });
}

function updateCharts() {
  // Update charts with real data
  // This would typically update the chart data based on allEnquiries
}

/* -------------------------
   Auto-assignment
   ------------------------- */
function autoAssignNewEnquiries(newEnquiries) {
  newEnquiries.forEach(enquiry => {
    if (!assignedEnquiries[enquiry.id]) {
      const agent = getNextAvailableAgent(enquiry);
      if (agent) {
        assignedEnquiries[enquiry.id] = agent.name;
        localStorage.setItem('assignedEnquiries', JSON.stringify(assignedEnquiries));
        
        // Send Telegram notification
        const message = formatTelegramMessage(enquiry, agent);
        sendTelegramMessage(agent.telegramId, message);
        
        showBootstrapToast(`Auto-assigned to ${agent.name}`, 'info');
      }
    }
  });
}

function getNextAvailableAgent(enquiry) {
  // Simple round-robin assignment
  const agent = agents[currentAgentIndex];
  currentAgentIndex = (currentAgentIndex + 1) % agents.length;
  return agent;
}

function formatTelegramMessage(enquiry, agent) {
  return `🚀 *New Client Enquiry Assigned*\n\n` +
    `👤 *Name:* ${enquiry.name || 'N/A'}\n` +
    `📞 *Phone:* ${enquiry.phone || 'N/A'}\n` +
    `🛠️ *Service:* ${enquiry.service || 'N/A'}\n` +
    `📋 *Message:* ${enquiry.requirement || 'N/A'}\n` +
    `📅 *Received:* ${new Date(enquiry.created_at).toLocaleString()}\n` +
    `👨‍💼 *Assigned To:* ${agent.name}\n\n` +
    `💡 *Please follow up within 10 Minutes*`;
}

function formatAgentMessage(enquiry, agentName, customMessage = '') {
  return `🚀 *New Client Enquiry Assigned to ${agentName}*\n\n` +
    `👤 *Name:* ${enquiry.name || 'N/A'}\n` +
    `📧 *Email:* ${enquiry.email || 'N/A'}\n` +
    `📞 *Phone:* ${enquiry.phone || 'N/A'}\n` +
    `📋 *Message:* ${enquiry.requirement || 'N/A'}\n` +
    `📅 *Received:* ${new Date(enquiry.created_at).toLocaleString()}\n` +
    `👨‍💼 *Assigned By:* Supervisor\n\n` +
    `💡 *Please follow up within 10 Minutes*` +
    (customMessage ? `\n\n*Additional Notes:* ${customMessage}` : '');
}

/* -------------------------
   Event Handlers
   ------------------------- */
$(document).ready(function() {
  // Initialize
  loadSavedContacts();
  initializeCharts();
  
  // Basic events
  $('#manualRefresh').click(fetchEnquiries);
  $('#connectTelegram').click(() => window.open('https://web.telegram.org/', '_blank'));
  $('#toggleMode').click(() => $('body').toggleClass('dark-mode'));

  // PDF generation
  $('#generatePDF').click(() => {
    const filename = `enquiry-report_${new Date().toISOString().slice(0,10)}.pdf`;
    generatePDF(allEnquiries, filename);
    showBootstrapToast('PDF generated with company header!', 'success');
  });

  // Telegram send
  $('#sendTelegramBtn').click(async () => {
    const chatId = $('#telegramChatId').val().trim() || $('#telegramContact').val();
    const fromDate = $('#fromDate').val(), toDate = $('#toDate').val();
    
    if(!chatId){ alert('Please select a contact or enter Chat ID'); return; }
    if(!fromDate || !toDate){ alert('Please select date range'); return; }
    
    const filtered = allEnquiries.filter(e => {
      const d = new Date(e.created_at);
      return d >= new Date(fromDate) && d <= new Date(toDate);
    });
    
    const message = `📊 *Enquiry Report*\nFrom: ${fromDate}\nTo: ${toDate}\nTotal: ${filtered.length} enquiries`;
    
    const success = await sendTelegramMessage(chatId, message);
    if (success) {
      showBootstrapToast('Report sent via Telegram!', 'success');
      const modal = bootstrap.Modal.getInstance(document.getElementById('sendTelegramModal'));
      modal.hide();
    } else {
      showBootstrapToast('Failed to send Telegram message', 'danger');
    }
  });

  // Filters
  $('#nameFilter, #phoneFilter, #dateFilter, #statusFilter, #serviceFilter').on('input change', function() {
    applyFilters();
    renderTable();
  });

  $('#rowsPerPage').change(function() {
    perPage = parseInt($(this).val());
    currentPage = 1;
    renderTable();
  });

  // Copy message functionality - FIXED
  $(document).on('click', '.copy-message-btn', function() {
    const message = $(this).data('message');
    navigator.clipboard.writeText(message).then(() => {
      showBootstrapToast('Message copied to clipboard!', 'success');
    }).catch(() => {
      const textArea = document.createElement('textarea');
      textArea.value = message;
      document.body.appendChild(textArea);
      textArea.select();
      document.execCommand('copy');
      document.body.removeChild(textArea);
      showBootstrapToast('Message copied to clipboard!', 'success');
    });
  });

  // Assign agent
  $(document).on('click', '.assign-agent-btn', function() {
    const enquiryId = $(this).data('enquiry-id');
    currentEnquiry = allEnquiries.find(e => e.id == enquiryId);
    
    if (currentEnquiry) {
      $('#enquiryDetails').val(
        `New Enquiry Received:\n\n` +
        `Name: ${currentEnquiry.name}\n` +
        `Email: ${currentEnquiry.email}\n` +
        `Phone: ${currentEnquiry.phone}\n` +
        `Service: ${currentEnquiry.service}\n` +
        `Message: ${currentEnquiry.requirement}\n` +
        `Date: ${new Date(currentEnquiry.created_at).toLocaleString()}`
      );
      $('#customMessage').val('');
      $('#agentSelect').val('');
      
      const modal = new bootstrap.Modal(document.getElementById('assignAgentModal'));
      modal.show();
    }
  });

  // Save contact
  $('#saveContactBtn').click(function() {
    const form = document.getElementById('addContactForm');
    if (!form.checkValidity()) {
      form.reportValidity();
      return;
    }
    
    const contact = {
      name: $('input[name="contactName"]').val(),
      designation: $('input[name="contactDesignation"]').val(),
      phone: $('input[name="contactPhone"]').val(),
      telegramChatId: $('input[name="telegramChatId"]').val()
    };
    
    saveContact(contact);
    const modal = bootstrap.Modal.getInstance(document.getElementById('addContactModal'));
    modal.hide();
    form.reset();
  });

  // Delete contact
  $(document).on('click', '.delete-contact', function(e) {
    e.stopPropagation();
    const index = $(this).data('index');
    if (confirm('Are you sure you want to delete this contact?')) {
      deleteContact(index);
    }
  });

  // Select contact
  $(document).on('click', '.saved-contact-item', function() {
    const index = $(this).data('index');
    const contact = savedContacts[index];
    $('#telegramChatId').val(contact.telegramChatId || '');
    const modal = bootstrap.Modal.getInstance(document.getElementById('savedContactsModal'));
    modal.hide();
  });

  // Auto-assign toggle
  $('#autoAssignBtn').click(function() {
    autoAssignEnabled = !autoAssignEnabled;
    if (autoAssignEnabled) {
      $(this).removeClass('btn-warning').addClass('btn-success');
      showBootstrapToast('Auto-assign enabled. New enquiries will be automatically assigned.', 'success');
    } else {
      $(this).removeClass('btn-success').addClass('btn-warning');
      showBootstrapToast('Auto-assign disabled.', 'warning');
    }
  });

  // Advanced reporting
  $('#generateAdvancedReport').click(function() {
    const reportType = $('#reportType').val();
    const fromDate = $('#reportFromDate').val();
    const toDate = $('#reportToDate').val();
    const format = $('#exportFormat').val();
    
    if (!fromDate || !toDate) {
      alert('Please select date range');
      return;
    }
    
    const filtered = allEnquiries.filter(e => {
      const d = new Date(e.created_at);
      return d >= new Date(fromDate) && d <= new Date(toDate);
    });
    
    const filename = `advanced-report_${fromDate}_to_${toDate}`;
    
    switch(format) {
      case 'pdf':
        generatePDF(filtered, filename);
        break;
      case 'excel':
        exportToExcel(filtered, filename);
        break;
      case 'csv':
        exportToCSV(filtered, filename);
        break;
    }
    
    showBootstrapToast(`Advanced ${format.toUpperCase()} report generated!`, 'success');
  });

  // Save automation settings
  $('#saveAutomationSettings').click(function() {
    automationSettings = {
      serviceAssignment: $('#serviceAssignment').is(':checked'),
      timeAssignment: $('#timeAssignment').is(':checked'),
      loadBalancing: $('#loadBalancing').is(':checked'),
      notificationSchedule: $('#notificationSchedule').val(),
      slaTracking: $('#slaTracking').is(':checked'),
      followupReminders: $('#followupReminders').is(':checked')
    };
    
    localStorage.setItem('automationSettings', JSON.stringify(automationSettings));
    showBootstrapToast('Automation settings saved!', 'success');
  });

  // Load saved settings
  if (Object.keys(automationSettings).length > 0) {
    $('#serviceAssignment').prop('checked', automationSettings.serviceAssignment);
    $('#timeAssignment').prop('checked', automationSettings.timeAssignment);
    $('#loadBalancing').prop('checked', automationSettings.loadBalancing);
    $('#notificationSchedule').val(automationSettings.notificationSchedule);
    $('#slaTracking').prop('checked', automationSettings.slaTracking);
    $('#followupReminders').prop('checked', automationSettings.followupReminders);
  }
});

function escapeHtml(text){
  if(!text && text !== 0) return '';
  return String(text).replace(/[&<>"'`=\/]/g, function(s) {
    return ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;','/':'&#47;','=':'&#61;','`':'&#96;'})[s];
  });
}

/* -------------------------
   Init / Polling
   ------------------------- */
fetchEnquiries();
setInterval(fetchEnquiries, 10000);
</script>