<?php
// index.php - complete frontend with advanced UI/UX for Token Management System
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Token Management System</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<script src="https://cdn.tailwindcss.com"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/qrcodejs/1.0.0/qrcode.min.js"></script>
<style>
/* Animations */
.fade-in { animation: fadeIn 0.6s ease-in; }
@keyframes fadeIn { from {opacity:0;} to {opacity:1;} }
.pulse { animation: pulse 1s infinite; }
@keyframes pulse {0%,100%{transform:scale(1);opacity:1;}50%{transform:scale(1.2);opacity:0.8;}}

/* Badges */
.badge { padding:0.25rem 0.5rem; border-radius:0.375rem; font-weight:600; color:white; }
.badge.NORMAL { background-color:#3B82F6; }
.badge.HIGH { background-color:#F59E0B; }
.badge.URGENT { background-color:#EF4444; }

/* Card */
.card { padding:1rem; border-radius:0.5rem; background:white; box-shadow:0 1px 3px rgba(0,0,0,0.1); transition:0.3s; }
.card:hover { box-shadow:0 4px 6px rgba(0,0,0,0.15); }

/* Ticker */
.ticker { white-space: nowrap; overflow: hidden; background:#E5E7EB; padding:0.5rem; border-radius:0.25rem; }
.ticker span { display:inline-block; padding-right:2rem; animation:ticker 15s linear infinite; }
@keyframes ticker {0%{transform:translateX(100%);}100%{transform:translateX(-100%);}}

/* Button */
button { transition: all 0.2s; }
button:hover { opacity:0.85; }

/* Responsive */
@media(max-width:640px){#adminPanel,#visitorPanel,#tvDisplay{padding:0.5rem;} #queue .card{flex-direction:column; align-items:flex-start;}}
</style>
</head>
<body class="bg-gray-100 min-h-screen font-sans">
<div id="app" class="p-4 max-w-6xl mx-auto">

<header class="flex justify-between items-center mb-4 flex-wrap">
<h1 class="text-2xl font-bold">🎫 Token Management System</h1>
<span id="status" class="text-sm px-3 py-1 rounded bg-gray-300">Checking...</span>
</header>

<nav class="flex space-x-2 mb-4 flex-wrap">
<button onclick="switchView('admin')" class="px-4 py-2 rounded bg-blue-500 text-white">Admin</button>
<button onclick="switchView('visitor')" class="px-4 py-2 rounded bg-green-500 text-white">Visitor</button>
<button onclick="switchView('display')" class="px-4 py-2 rounded bg-purple-500 text-white">TV Display</button>
</nav>

<!-- Admin Panel -->
<section id="adminPanel" class="hidden space-y-4">
<div class="card">
<h2 class="font-semibold mb-2">Issue Token</h2>
<input id="visitorName" placeholder="Visitor Name" class="border p-2 w-full mb-2 rounded">
<input id="visitPurpose" placeholder="Purpose of Visit" class="border p-2 w-full mb-2 rounded">
<input id="visitorPhone" placeholder="Phone (optional)" class="border p-2 w-full mb-2 rounded">
<select id="priority" class="border p-2 w-full mb-2 rounded">
<option value="NORMAL">Normal</option>
<option value="HIGH">High</option>
<option value="URGENT">Urgent</option>
</select>
<button onclick="issueToken()" class="px-4 py-2 bg-blue-600 text-white rounded">Create Token</button>
</div>

<div class="card">
<h2 class="font-semibold mb-2">Today's Queue</h2>
<div id="queue" class="space-y-2"></div>
</div>

<div class="card">
<h2 class="font-semibold mb-2">Stats</h2>
<div id="stats"></div>
</div>
</section>

<!-- Visitor Panel -->
<section id="visitorPanel" class="hidden text-center space-y-4">
<div class="card">
<h2 class="text-xl font-bold mb-4">Visitor Queue Status</h2>
<p>Now Serving: <span id="visitorNowServing" class="font-bold text-2xl"></span></p>
<p>Your Position: <span id="visitorPosition" class="font-semibold"></span></p>
<p>Estimated Wait: <span id="visitorWait" class="font-semibold"></span></p>
<div id="visitorQR" class="mx-auto mt-4"></div>
</div>
</section>

<!-- TV Display -->
<section id="tvDisplay" class="hidden text-center">
<h1 class="text-5xl font-bold mb-6">Now Serving</h1>
<div id="currentToken" class="text-8xl font-extrabold text-blue-600 fade-in pulse"></div>
<div id="recentTokens" class="mt-6 text-3xl text-gray-600"></div>
<div class="ticker mt-6 py-2"><span id="tickerContent"></span></div>
<audio id="audioAlert" src="https://actions.google.com/sounds/v1/alarms/beep_short.ogg"></audio>
</section>

<script>
let apiBase = "https://targetmaids.ae/public/targetapps/token/api.php";
let refreshInterval = 5000;

// ===== API Wrapper =====
async function api(path, opts={}) {
try {
const res = await fetch(`${apiBase}?path=${path}`, opts);
if(!res.ok) throw new Error(res.status);
const data = await res.json();
document.getElementById("status").innerText="Online ✅";
document.getElementById("status").className="bg-green-200 px-3 py-1 rounded";
return data;
} catch(e) {
document.getElementById("status").innerText="Offline ❌";
document.getElementById("status").className="bg-red-200 px-3 py-1 rounded";
throw e;
}
}

// ===== Issue Token =====
async function issueToken(){
const visitor_name = document.getElementById("visitorName").value;
const visit_purpose = document.getElementById("visitPurpose").value;
const phone = document.getElementById("visitorPhone").value;
const priority = document.getElementById("priority").value;
if(!visitor_name || !visit_purpose){alert("Name & Purpose required!");return;}
try{
const res = await api("tokens",{method:"POST",headers:{"Content-Type":"application/x-www-form-urlencoded"},body:new URLSearchParams({visitor_name,visit_purpose,phone,priority})});
if(res.status==="success"){
alert(`Token #${res.data.token_number} created`);
document.getElementById("visitorName").value="";
document.getElementById("visitPurpose").value="";
document.getElementById("visitorPhone").value="";
loadQueue();
}
}catch(e){alert("Failed to create token");}
}

// ===== Load Queue =====
let lastCurrentToken=null;
async function loadQueue(){
try{
const res = await api("today");
if(res.status==="success") renderQueue(res.data);
}catch(e){}
}

function renderQueue(tokens){
const qDiv=document.getElementById("queue");
qDiv.innerHTML="";
tokens.forEach(t=>{
const el=document.createElement("div");
el.className="card flex justify-between items-center";
el.innerHTML=`<div><span class="font-bold text-lg">#${t.token_number}</span> ${t.visitor_name} <span class="badge ${t.priority}">${t.priority}</span> <span class="text-gray-500 ml-2">${t.status}</span></div>
<div class="space-x-2 mt-2 sm:mt-0">
<button onclick="updateStatus(${t.id},'IN_PROGRESS')" class="px-2 py-1 bg-yellow-500 text-white rounded">▶ Start</button>
<button onclick="updateStatus(${t.id},'COMPLETED')" class="px-2 py-1 bg-green-600 text-white rounded">✔ Done</button>
</div>`;
qDiv.appendChild(el);
});

// Stats
document.getElementById("stats").innerHTML=`Total: ${tokens.length}<br>Pending: ${tokens.filter(t=>t.status==="PENDING").length}<br>In Progress: ${tokens.filter(t=>t.status==="IN_PROGRESS").length}<br>Completed: ${tokens.filter(t=>t.status==="COMPLETED").length}`;

// Visitor & TV
const current=tokens.find(t=>t.status==="IN_PROGRESS");
const recents=tokens.filter(t=>t.status==="COMPLETED").slice(-3);
document.getElementById("currentToken").innerText=current?`#${current.token_number}`:"--";
document.getElementById("recentTokens").innerText=recents.map(t=>"#"+t.token_number).join(" • ");
document.getElementById("tickerContent").innerText=tokens.filter(t=>t.status==="PENDING").map(t=>"#"+t.token_number).join(" • ");

// Audio alert for new current token
if(current && current.token_number !== lastCurrentToken){
document.getElementById("audioAlert").play();
lastCurrentToken=current.token_number;
}

// Visitor
document.getElementById("visitorNowServing").innerText=current?`#${current.token_number}`:"None";
const pending=tokens.filter(t=>t.status==="PENDING");
document.getElementById("visitorPosition").innerText=pending.length?`~${pending.length} before you`:"You're next!";
document.getElementById("visitorWait").innerText=pending.length?`${pending.length*2} mins`:"0 mins";

// Generate QR for visitor
const qrDiv=document.getElementById("visitorQR");
qrDiv.innerHTML="";
if(current){new QRCode(qrDiv,{text:`Token #${current.token_number}\nName: ${current.visitor_name}`,width:128,height:128});}
}

// ===== Update Status =====
async function updateStatus(id,status){
try{
await api("tokens",{method:"POST",headers:{"Content-Type":"application/x-www-form-urlencoded"},body:new URLSearchParams({action:"update",id,status})});
loadQueue();
}catch(e){alert("Update failed");}
}

// ===== Auto Refresh =====
async function autoRefresh(){await loadQueue();setTimeout(autoRefresh,refreshInterval);}

// ===== Switch View =====
function switchView(view){
["adminPanel","visitorPanel","tvDisplay"].forEach(id=>document.getElementById(id).classList.add("hidden"));
if(view==="admin") document.getElementById("adminPanel").classList.remove("hidden");
if(view==="visitor") document.getElementById("visitorPanel").classList.remove("hidden");
if(view==="display") document.getElementById("tvDisplay").classList.remove("hidden");
}

// ===== Init =====
switchView("admin");
autoRefresh();
</script>
</body>
</html>
