// Initialize IndexedDB
let db;
const request = indexedDB.open("AttendanceDB", 1);

// Handle errors while opening the database
request.onerror = function(event) {
    console.error("Database error: ", event.target.errorCode);
};

// On successful database opening
request.onsuccess = function(event) {
    db = event.target.result;
    console.log("Database opened successfully.");
    loadAllEmployees(); // Load employees after database is ready
};

// Upgrade needed to create the object store
request.onupgradeneeded = function(event) {
    db = event.target.result;
    const objectStore = db.createObjectStore("attendance", { keyPath: "id" }); // Create an object store
    console.log("Object store created.");
};

// List of employees
const employees = [
    "Azizah", "Eled", "Hessenet", "Timaj", "Maryam (Cashier)", "Maryam (Typing)", 
    "Queen", "Florence", "Joan", "Grace", "Kedija Jamie", "Reem Khalid", "Amal Khaled", 
    "Alganesh", "Aya Nasreldin", "Muna Zeku", "Analie Atilano", "Fatima Ali", 
    "Wesam", "Abdul Salam", "Azeb Muhammad", "Uzair", "Lilly", "Faheem", "Queen"
];

// Current date for attendance tracking
let currentDate = new Date().toISOString().split('T')[0]; // Format: YYYY-MM-DD

// Load all employees into the table
function loadAllEmployees() {
    let tableBody = $('#attendanceRecords');
    tableBody.empty(); // Clear existing records

    employees.forEach((name) => {
        const todayRecord = { date: currentDate, checkIn: '', checkOut: '', status: 'Absent' }; // Default record

        // Retrieve data from IndexedDB
        const transaction = db.transaction(["attendance"], "readonly");
        const objectStore = transaction.objectStore("attendance");
        const request = objectStore.get(`${name}-${currentDate}`);

        request.onsuccess = function(event) {
            const record = event.target.result;

            // If record exists, populate today's record with stored data
            if (record) {
                todayRecord.checkIn = record.checkIn;
                todayRecord.checkOut = record.checkOut;
                todayRecord.status = record.status;
            }

            let row = `<tr>
                <td><input type="text" value="${name}" readonly></td>
                <td><input type="text" value="${todayRecord.checkIn}" readonly></td>
                <td><input type="text" value="${todayRecord.checkOut}" readonly></td>
                <td><input type="text" value="${todayRecord.status}" readonly></td>
                <td class="action-buttons">
                    <button type="button" class="checkInBtn btn btn-success">Check-in</button>
                    <button type="button" class="checkOutBtn btn btn-primary" ${todayRecord.checkIn ? '' : 'disabled'}>Check-out</button>
                </td>
            </tr>`;
            tableBody.append(row);
        };

        request.onerror = function(event) {
            console.error("Error retrieving record:", event.target.errorCode);
        };
    });
}

// Update summary counts
function updateSummary() {
    const totalEmployees = employees.length;
    let presentCount = 0;
    let absentList = [];

    employees.forEach(name => {
        const transaction = db.transaction(["attendance"], "readonly");
        const objectStore = transaction.objectStore("attendance");
        const request = objectStore.get(`${name}-${currentDate}`);

        request.onsuccess = function(event) {
            const record = event.target.result;

            // If record exists and status is 'Present'
            if (record && record.status === 'Present') {
                presentCount++;
            } else {
                absentList.push(name);
            }

            // Update summary after checking all employees
            $('#summary').text(`Total Employees: ${totalEmployees}, Present: ${presentCount}, Absent: ${totalEmployees - presentCount}`);
            $('#absentList').empty();
            absentList.forEach(name => {
                $('#absentList').append(`<li>${name}</li>`);
            });
        };
    });
}

// Save attendance to IndexedDB
function saveAttendanceToDB(attendance) {
    const transaction = db.transaction(["attendance"], "readwrite");
    const objectStore = transaction.objectStore("attendance");
    
    attendance.forEach(record => {
        objectStore.put(record); // Insert or update record
    });

    transaction.oncomplete = function() {
        console.log("Attendance data saved to IndexedDB");
    };

    transaction.onerror = function(event) {
        console.error("Error saving to IndexedDB: ", event.target.errorCode);
    };
}

// Function to handle Check-in button click
$(document).on('click', '.checkInBtn', function() {
    const row = $(this).closest('tr');
    const name = row.find('input[type="text"]').val();
    const checkInTime = new Date().toLocaleTimeString();
    
    // Update row with check-in time and status
    row.find('input[type="text"]').eq(1).val(checkInTime);
    row.find('.checkOutBtn').prop('disabled', false);
    row.find('input[type="text"]').eq(3).val('Present');

    // Create a record for attendance
    const record = {
        id: `${name}-${currentDate}`, // Unique ID for each record
        name: name,
        date: currentDate,
        checkIn: checkInTime,
        checkOut: '',
        status: 'Present'
    };

    // Save to IndexedDB
    saveAttendanceToDB([record]); // Save the new record
    updateSummary(); // Update summary after check-in
});

// Function to handle Check-out button click
$(document).on('click', '.checkOutBtn', function() {
    const row = $(this).closest('tr');
    const name = row.find('input[type="text"]').val();
    const checkOutTime = new Date().toLocaleTimeString();
    
    // Update the row with check-out time
    row.find('input[type="text"]').eq(2).val(checkOutTime);
    
    // Update the latest attendance record for check-out
    const latestRecord = {
        id: `${name}-${currentDate}`, // Unique ID
        name: name,
        date: currentDate,
        checkIn: row.find('input[type="text"]').eq(1).val(),
        checkOut: checkOutTime,
        status: 'Present' // Update status if needed
    };

    // Save updated attendance data to IndexedDB
    saveAttendanceToDB([latestRecord]); // Save the updated record
    console.log("Updated attendance data on checkout:", latestRecord); // Debugging
});

// Search functionality
function searchEmployeeAttendance() {
    const searchTerm = $('#searchInput').val().toLowerCase();
    const selectedDate = $('#datepicker').val() || currentDate; // Use selected date or current date if none
    const filteredEntries = [];

    employees.forEach(name => {
        const transaction = db.transaction(["attendance"], "readonly");
        const objectStore = transaction.objectStore("attendance");
        const request = objectStore.get(`${name}-${selectedDate}`);

        request.onsuccess = function(event) {
            const entry = event.target.result;
            if (entry && name.toLowerCase().includes(searchTerm)) {
                filteredEntries.push({
                    name: entry.name,
                    checkIn: entry.checkIn,
                    checkOut: entry.checkOut,
                    status: entry.status,
                    date: entry.date // Add date for display
                });
            }

            // Populate the table with search results once all requests are done
            if (filteredEntries.length === employees.length) {
                populateTable(filteredEntries);
            }
        };

        request.onerror = function(event) {
            console.error("Error retrieving attendance data:", event.target.errorCode);
        };
    });
}

// Function to populate table based on search results
function populateTable(data) {
    let tableBody = $('#attendanceRecords');
    tableBody.empty(); // Clear existing rows

    data.forEach((entry) => {
        let row = `<tr>
            <td><input type="text" value="${entry.name}" readonly></td>
            <td><input type="text" value="${entry.checkIn}" readonly></td>
            <td><input type="text" value="${entry.checkOut}" readonly></td>
            <td><input type="text" value="${entry.status}" readonly></td>
            <td class="action-buttons">
                <button type="button" class="checkInBtn btn btn-success">Check-in</button>
                <button type="button" class="checkOutBtn btn btn-primary" ${entry.checkOut ? '' : 'disabled'}>Check-out</button>
            </td>
        </tr>`;
        tableBody.append(row);
    });
}

// Submit attendance functionality (optional)
$('#submitAttendance').on('click', function() {
    alert('Attendance submitted successfully!');
});

// Datepicker initialization
$('#datepicker').datepicker({
    format: 'yyyy-mm-dd',
    autoclose: true,
    todayHighlight: true
}).datepicker('setDate', new Date()).on('changeDate', function() {
    currentDate = $(this).val(); // Update current date
    loadAllEmployees(); // Reload employee attendance for the new date
});

// Load all employees on page load
loadAllEmployees();

// PDF Download (optional)
$('#downloadPDF').on('click', function () {
    const element = document.body;
    const options = {
        margin: 0.5,
        filename: 'attendance_summary.pdf',
        image: { type: 'jpeg', quality: 0.98 },
        html2canvas: { scale: 2 },
        jsPDF: { unit: 'in', format: 'letter', orientation: 'portrait' }
    };
    html2pdf().from(element).set(options).save();
});
